/*
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 * UID.h
 * Representation of an RDM UID
 * Copyright (C) 2005-2010 Simon Newton
 */

#ifndef INCLUDE_OLA_RDM_UID_H_
#define INCLUDE_OLA_RDM_UID_H_

#include <stdint.h>
#include <iomanip>
#include <sstream>
#include <string>

namespace ola {
namespace rdm {

using std::ostream;
using std::string;


/*
 * Represents a RDM UID.
 */
class UID {
  public:
    UID(uint16_t esta_id, uint32_t device_id) {
      m_uid.esta_id = esta_id;
      m_uid.device_id = device_id;
    }

    UID& operator=(const UID& other) {
      if (this != &other) {
        m_uid.esta_id = other.m_uid.esta_id;
        m_uid.device_id = other.m_uid.device_id;
      }
      return *this;
    }

    explicit UID(const uint8_t *data) {
      m_uid.esta_id = static_cast<uint16_t>((data[0] << 8) + data[1]);
      m_uid.device_id = static_cast<uint32_t>(data[2] << 24) +
                        static_cast<uint32_t>(data[3] << 16) +
                        static_cast<uint32_t>(data[4] << 8) +
                        data[5];
    }

    bool operator==(const UID &other) const {
      return 0 == cmp(*this, other);
    }

    bool operator!=(const UID &other) const {
      return 0 != cmp(*this, other);
    }

    bool operator>(const UID &other) const {
      return cmp(*this, other) > 0;
    }

    bool operator<(const UID &other) const {
      return cmp(*this, other) < 0;
    }

    uint16_t ManufacturerId() const { return m_uid.esta_id; }

    uint32_t DeviceId() const { return m_uid.device_id; }

    bool IsBroadcast() const { return m_uid.device_id == ALL_DEVICES; }

    /**
     * Returns true if the uid in the argument should react to a command
     */
    bool DirectedToUID(const UID &uid) const {
      return *this == uid ||
            (IsBroadcast() &&
             (ManufacturerId() == ALL_MANUFACTURERS ||
              ManufacturerId() == uid.ManufacturerId()));
    }

    std::string ToString() const {
      std::stringstream str;
      str << std::setfill('0') << std::setw(4) << std::hex << m_uid.esta_id
        << ":" << std::setw(8) << m_uid.device_id;
      return str.str();
    }

    friend ostream& operator<< (ostream &out, const UID &uid) {
      return out << uid.ToString();
    }

    bool Pack(uint8_t *buffer, unsigned int length) const {
      if (length < UID_SIZE)
        return false;
      buffer[0] = static_cast<uint8_t>(m_uid.esta_id >> 8);
      buffer[1] = static_cast<uint8_t>(m_uid.esta_id & 0xff);
      buffer[2] = static_cast<uint8_t>(m_uid.device_id >> 24);
      buffer[3] = static_cast<uint8_t>(m_uid.device_id >> 16);
      buffer[4] = static_cast<uint8_t>(m_uid.device_id >> 8);
      buffer[5] = static_cast<uint8_t>(m_uid.device_id & 0xff);
      return true;
    }

    static UID AllDevices() {
      return UID(ALL_MANUFACTURERS, ALL_DEVICES);
    }

    static UID AllManufactureDevices(uint16_t esta_id) {
      return UID(esta_id, ALL_DEVICES);
    }

    static UID* FromString(const string &uid);

    enum { UID_SIZE = 6 };

    static const uint16_t ALL_MANUFACTURERS = 0xffff;
    static const uint32_t ALL_DEVICES = 0xffffffff;

  private:
    struct rdm_uid {
      uint16_t esta_id;
      uint32_t device_id;
    };

    struct rdm_uid m_uid;

    int cmp(const UID &a, const UID &b) const {
      if (a.m_uid.esta_id == b.m_uid.esta_id)
        return cmp(a.m_uid.device_id, b.m_uid.device_id);
      return cmp(a.m_uid.esta_id, b.m_uid.esta_id);
    }

    int cmp(uint32_t a, uint32_t b) const {
      if (a == b)
        return 0;
      return a < b ? -1 : 1;
    }
};
}  // rdm
}  // ola
#endif  // INCLUDE_OLA_RDM_UID_H_
